﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using NovelInterpreter.Exception;
using NovelInterpreter.Instance.Drawer;

using UnityEngine;

namespace NovelInterpreter.Instance.Utility
{
	/// <summary>
	/// ウィンドウの設定
	/// </summary>
	public class TextureUtility
	{
		/// <summary>
		/// 相対座標を求める
		/// </summary>
		/// <param name="frame">ウィンドウの縦か横か</param>
		/// <param name="pos">XかYか</param>
		/// <returns>相対座標</returns>
		static float CalcRelation(int frame, int pos)
		{
			return (1f / frame) * pos;
		}

		/// <summary>
		/// 絶対座標を求める
		/// </summary>
		/// <param name="frame">ウィンドウの縦か横か</param>
		/// <param name="pos">XかYか</param>
		/// <returns>絶対座標</returns>
		static int CalcAbsolute(float frame, float pos)
		{
			return (int)(frame * pos);
		}

		/// <summary>
		/// 水平垂直問わずセンタリング
		/// </summary>
		/// <param name="textureLength">テクスチャの長さ（縦か横か）</param>
		/// <param name="scaleSize">対応したスケーリング</param>
		/// <param name="targetPosition">対応した移動したい位置</param>
		/// <param name="targetScreen">対応したスクリーンの長さ</param>
		/// <returns>センタリングされた位置</returns>
		static float Centering(float textureLength, float scaleSize, float targetPosition, float targetScreen)
		{
			float width = textureLength * 0.5f * scaleSize;
			targetPosition = targetScreen * 0.5f - width;
			targetPosition *= 1f / targetScreen;
			return targetPosition;
		}

		/// <summary>
		/// 垂直方向にセンタリングする
		/// </summary>
		/// <param name="drawer">対象のウィンドウ</param>
		public static void CenteringVertical(WindowDrawer drawer, Transform transform)
		{
			Vector3 pos = transform.position;
			pos.y = Centering(drawer.Texture.height, transform.localScale.y, transform.position.y, Screen.height);
			transform.position = pos;
		}

		/// <summary>
		/// 水平方向にセンタリングする
		/// </summary>
		/// <param name="drawer">対象のウィンドウ</param>
		public static void CenteringHorizontal(WindowDrawer drawer, Transform transform)
		{
			Vector3 pos = transform.position;
			pos.x = Centering(drawer.Texture.width, transform.localScale.x, transform.position.x, Screen.width);
			transform.position = pos;
		}

		/// <summary>
		/// 水平方向にウィンドウを伸ばす
		/// 縦も一緒に伸びる
		/// </summary>
		/// <param name="drawer">対象のウィンドウ</param>
		/// <param name="percent">ウィンドウに対する比率</param>
		public static void StretchHorizontal(WindowDrawer drawer, Transform transform, float percent)
		{
			transform.localScale = StretchSize(
				transform.localScale, drawer.Texture.width, Screen.width, percent);
		}

		/// <summary>
		/// 垂直方向にウィンドウを伸ばす
		/// 横も一緒に伸びる
		/// </summary>
		/// <param name="drawer">対象のウィンドウ</param>
		/// <param name="percent">ウィンドウに対する比率</param>
		public static void StretchVertical(WindowDrawer drawer, Transform transform, float percent)
		{
			transform.localScale = StretchSize(
				transform.localScale, drawer.Texture.height, Screen.height, percent);
		}

		/// <summary>
		/// 適当な方向に伸ばしたものを返す
		/// </summary>
		/// <param name="windowScale">MessageWindowDrawerが持つスケール値</param>
		/// <param name="size">テクスチャのwidthもしくはheight</param>
		/// <param name="percent">画面に対する比率</param>
		/// <returns>伸ばしたスケール値</returns>
		static Vector3 StretchSize(Vector3 windowScale, float size, float targetWindowLength, float percent)
		{
			float width = size;
			float windowRatio = targetWindowLength / width;			// テクスチャがウィンドウに対してどれぐらいの比率か
			float pixelRatio = windowRatio / targetWindowLength;	// テクスチャを1ピクセルの大きさにする
			float scaleWidth = pixelRatio * percent;				// 1ピクセルの大きさを比率で拡大する

			windowScale.x = scaleWidth;
			windowScale.y = scaleWidth;
			return windowScale;
		}
	}
}
